require('dotenv').config(); // Load environment variables

const express = require('express');
const mysql = require('mysql2');
const cors  = require('cors');
const bodyParser = require('body-parser');
const africastalking = require('africastalking');

const app = express();
app.use(cors());
app.use(express.json());
app.use(bodyParser.json());

app.use((req, res, next) => {
  console.log(`➡️ ${req.method} ${req.url}`);
  next();
});

//connect to database
const db = mysql.createConnection({
    host: process.env.DB_HOST,
    user: process.env.DB_USER,
    password: process.env.DB_PASSWORD,
    database: process.env.DB_NAME,
});

// Check database connection
db.connect((err) =>{
    if (err) {
        console.error('Database Connection Failed:', err);
        return;
    }
    console.log('Connection Successful', db.threadId);
});


app.post('/api/login',  (req, res) => {
  const { email, password } = req.body;
  const sql = 'SELECT * FROM users WHERE email = ? AND password = ?';

  db.query(sql, [email, password], (err, results) => {
    if (err) return res.status(500).json({ success: false, error: 'Database error' });
    if (results.length > 0) {
      res.json({ success: true, user: results[0] });
    } else {
      res.json({ success: false });
    }
  });
});

//API endpoint to get all contacts
app.get('/contacts', (req, res) => {
    db.query('SELECT * FROM contacts', (err, results) => {
        if (err) {
            return res.status(500).json({ error: 'Database query failed' });
        }
        res.json(results);
    });
});

//API endpoint to add new contact
app.post('/api/insert', express.json(), (req, res) => {
    console.log('Received body:', req.body);
    const { name, contact, company} = req.body;
    if (!name || !contact) {
        return res.status(400).json({ error: 'Name and Contact are required' });
    }
    const sql = 'INSERT INTO contacts (name, contact, company) VALUES (?, ?, ?)';

    db.query(sql, [name, contact, company], (err, results) => {
        if (err) {
            console.error('Insert failed:', err);
            return res.status(500).json({ error: 'Database insert failed' });
        } else {
            res.send('Contact added successfully');
        }
    });
});

//Africastalking configuration
const at = africastalking({
    apiKey: process.env.AT_API_KEY, // Use your Africa's Talking API key
    username: process.env.AT_USERNAME // Use your Africa's Talking username,
});
const sms = at.SMS;

// Array to hold delivery reports
let deliveryReports = [];
const messageTracker = {};
const e = require('express');

//API endpoint to send SMS
app.post('/send-sms', express.json(), async (req, res) => {
    const { to, message } = req.body;    
    console.log('📥 Incoming SMS Request:', req.body);
    if (!to || !message) {
        return res.status(400).json({ error: 'To and Message are required' });
    }

    //Convert to array if multiple numbers are provided
    const toList = to.split(',').map(c => c.trim()).filter(c => c !== '');

    try{
        const response = await sms.send({
            to: toList,
            message: message,
            from: 'UKWELILANDS', // Replace with your sender ID
            enqueue: true, // Optional: enqueue the message for delivery
            callback: 'https://sms.groundapp.co.ke/api/dlr' // ⚠️ Replace with your public HTTPS URL (ngrok for local testing)
        });

        if(
            response &&
            response.SMSMessageData &&
            Array.isArray(response.SMSMessageData.Recipients)
        ){
            response.SMSMessageData.Recipients.forEach((recipient) =>{
                messageTracker[recipient.messageId] = message;
            });
        }else{
            console.error('Unexpected response format:', response);
        }

        console.log('SMS sent successfully:', response);
        res.json({ success: true, response });

    } catch(error) {
        console.error('SMS Error:', error);
        res.status(500).json({ success: false, message: 'Failed to send SMS', error: error.message });
    }
});



//Delivery report endpoint
app.post('/api/dlr', express.urlencoded({ extended: true }), (req, res) => {

    const dlr = req.body;

    console.log('📨 Delivery Report Received:', dlr);

    const messageId = dlr.id;
    const phoneNumber = dlr.phoneNumber;
    const status = dlr.status;
    const cost = dlr.cost || 0;

    if (!messageId || !phoneNumber || !status) {
      return res.sendStatus(400);
    }

    const message = messageTracker[messageId] || '[Original message not found]';

    const sql = 'INSERT INTO sms_outbox (text, contact, status, date, messageId, cost, company) VALUES (?, ?, ?, ?, ?, ?, ?)';
    db.query(sql, [message, phoneNumber, status, new Date(), messageId, cost, 'UKWELILANDS'], (err, results) => {
      if (err) {
        if(err.code === 'ER_DUP_ENTRY') {
          console.log('⚠️ Duplicate messageId received, skipping...');
          return res.sendStatus(200); // Ignore duplicates
        }
        console.error('❌ Failed to save delivery report:', err.sqlMessage);
        return res.sendStatus(500);
      }
      console.log('✅ Delivery report saved to DB');
      res.sendStatus(200); // Must return 200 or AT will retry
    });

        // Store the delivery report
    deliveryReports.push({
        phoneNumber: dlr.phoneNumber,
        status: dlr.status,
        messageId: dlr.messageId,
        timestamp: new Date().toISOString(),
        cost: dlr.cost,
    });


    
});

app.get('/api/delivery-reports/by-ids', (req, res) => {
    const ids = req.query.ids;

    if (!ids) {
        return res.status(400).json({ error: 'Missing ids query parameter' });
    }

    const idArray = ids.split(',').map(id => id.trim());



    const sql = `SELECT * FROM sms_outbox WHERE messageId IN (${idArray.map(() => '?').join(',')})`;

    db.query(sql, idArray, (err, results) => {
        if (err) {
            console.error('❌ Failed to fetch delivery reports:', err);
            return res.status(500).json({ error: 'Database query failed' });
        }

        // Return the individual reports
        res.json({
            success: true,
            reports: results
        });
    });
});


//API endpoint to get delivery reports
app.get('/api/delivery-reports', (req, res) => {
  const sql = 'SELECT * FROM sms_outbox ORDER BY date DESC';

  db.query(sql, (err, results) => {
    if (err) {
      console.error('❌ Failed to fetch delivery reports:', err.sqlMessage);
      return res.status(500).json({ success: false, error: 'Database error' });
    }

    const summary = {
      totalSent: results.length,
      delivered: 0,
      failed: 0,
      pending: 0,
    };

    results.forEach(report => {
      const status = report.status?.toLowerCase() || '';
      if (status === 'delivered' || status === 'success') {
        summary.delivered++;
      } else if (status === 'failed') {
        summary.failed++;
      } else if (status === 'pending') {
        summary.pending++;
      }
    });

    res.json({
      success: true,
      summary,
      reports: results,
    });
  });
});

// Serve frontend build folder
const path = require('path');
app.use(express.static(path.join(__dirname, 'build')));

// Catch-all route: React handles the routing
app.get('*', (req, res) => {
  res.sendFile(path.join(__dirname, 'build', 'index.html'));
});

app.listen(process.env.PORT, () => {
    console.log(`Server is running on port ${process.env.PORT || 5000}`); // Use PORT from .env or default to 5000
});

